"abstract classes for deep models"

import inspect
import os
import logging
import abc
from operator import itemgetter, attrgetter

import numpy as np
import pandas as pd
import theano

from . import config_spec
from .. import archive

#__all__ = ["config_spec", "nccn", "train", "monitor"]

__spec_cls__ = lambda t: (t[0].endswith("Spec") and inspect.isclass(t[1]))
spec_list = map(itemgetter(1),
                filter(__spec_cls__, inspect.getmembers(config_spec)))


log = logging.getLogger(__name__)


class Layer( object ):
    """abstract layer class.

    A layer is a list of weights. Weights are
    theano shared tensors that are needed for example to compute activation.
    You can use them also for other reasons e.g., speeds to implement
    momentum GSD.
    """

    __metaclass__ = abc.ABCMeta

    def __init__(self, wshape_lst, wnames, fanin_lst, rng, wdtype):
        """initialize by the given weight properties, all params are lists
        of the same length

        :param wshape_lst: shapes of weights
        :param wnames: names of weights
        :param fanin_lst: fan in of weights
        :param rng: a random number generator or init value (see alloc_shared_weights_)
        :param wdtype: data type of weights.
        """

        if len(wshape_lst) != len(wnames):
            raise ValueError("length names and shapes differ")

        self._weights_ = []
        for (sh, n, fi, r, dt) in zip(wshape_lst, wnames, fanin_lst, rng, wdtype):
            self._weights_.append( alloc_shared_weights_(sh, dt, n, fi, r) )

    @abc.abstractmethod
    def get_params(self):
        """weights as symbolic variables"""
        pass

    @abc.abstractmethod
    def activation(self):
        "the symbolic activation function"
        pass

    def get_weights(self):
        """weights as ndarrays"""

        return map(lambda w: w.get_value(), self.get_params())

    @abc.abstractmethod
    def set_params(self, params):
        """set weights

        :param ndarray params: tuple of parameters
        :rtype: None"""

        pass

    def set_weights(self, vlst):
        """set the weight values of this layer"""

        assert len(vlst) == len(self.get_params())

        for v, sym_w in zip(vlst, self.get_params()):
            sym_w.set_value( v )

    def weight_norm(self, degree):
        """compute the L1 or L2 norm of weights

        :param degree: keyword `l1` or `l2`
        :rtype: the L1 or L2 norm of weights (float)"""

        f = {"l1": lambda w: np.abs( w ).sum(),
             "l2": lambda w: (np.abs( w ) ** 2).sum()}[degree]

        return sum( map(f, self.get_weights()) )

    def get_flat_weights(self):
        """get a list of the flattened version of the weights of this layer

        :rtype: list[ndarray]"""

        olst = []
        for w in self.get_weights():
            olst.append( w.reshape(-1,) )
        return olst

    def load_flat_weights(self, flatp):
        """reshape the given flattened weights into the correct dimensions
        for this layer

        :param flatp: flattened weights (list[ndarray])
        :rtype: the same given weights reshaped"""

        for i in range(len(flatp)):
            flatp[i] = flatp[i].reshape( self.get_weights()[i].shape )
        return flatp


class SpeedLayer( Layer ):
    """This layer provides an extra set of weights as a support
    the momentum algorithm for SGD.
    At time point t, we need weights at t-1 and the gradient at
    t to update weights. Namely

    w(t+1) - w(t) = - rho * dE(w)/dw + p s(t), for t = 0, 1, ... and s(0) = 0

    I call w(t) - w(t-1) = s(t) (speed)
    """

    #__metaclass__ = abc.ABCMeta

    def __init__(self, ws, wn, rng, wdtype):
        super(SpeedLayer, self).__init__(ws + ws,
                                         wn + map(lambda n: "%s_speed" % n, wn),
                                         map(sum, ws) + map(sum, ws),
                                         rng + [0.0] * len(rng),
                                         wdtype + wdtype)
        self.__wl = len(ws)

    def get_speeds(self):
        """get speeds

        :rtype: a tuple of (ndarray) speeds """

        return map(lambda i: self._weights_[self.__wl + i].get_value(),
                   range(self.__wl))

    def set_speeds(self, v):
        """set speeds

        :param v: tuple of (ndarray)
        :rtype: none"""

        for i in range(self.__wl):
            self._weights_[self.__wl + i].set_value( v[i] )

    def get_params(self):
        """get weights

        :rtype: a tuple of (theano.tensor) weights"""

        return self._weights_[0:self.__wl]

    def set_params(self, params):
        """set weights

        :param ndarray params: tuple of weights
        :rtype: None"""

        for i in range(len(self.__wl)):
            self._weights_[i] = params[i]

    @classmethod
    def _speed_update_f(cls, s, g, mu, rho):
        """speed update formula

        new_speed = -rho * g + mu*s

        :param s: cur speed
        :param g: gradient
        :param mu: momentum
        :param rho: learning rate
        :rtype: new speed"""

        return (np.array(s) * mu) - (rho * np.array(g))

    @classmethod
    def _weight_update_f(cls, cur_weight, speed):
        """speed update for the given gradient

        new_weight = cur_weight + speed

        :param cur_weight: current weight (ndarray)
        :param speed: speed (ndarray)
        :rtype: new_weight (ndarray)
        """

        return cur_weight + speed

    def speed_update(self, gradient, mu, rho):
        """update speeds for the given gradients

        new_speed = -rho * gradient + mu*cur_speed

        :param gradient: gradient (list of ndarray)
        :param mu: momentum (float)
        :param rho: learning rate (float)
        :rtype: new_speed (ndarray)
        """

        upd_f = self._speed_update_f
        new_spd = []
        for i in range(self.__wl):
            new_spd.append(upd_f(self.get_speeds()[i], gradient[i], mu, rho))
        self.set_speeds(new_spd)

    def weight_update(self):
        """update speeds for the current speed

        new_weight = cur_weight + cur_speed

        """
        upd_f = self._weight_update_f
        new_w = []
        for i in range(self.__wl):
            new_w.append( upd_f(self.get_weights()[i], self.get_speeds()[i]) )
        self.set_weights(new_w)

#    def iter_param_info(self):
#        layer_weights = self.get_flat_weights()
#        layer_weight_names = map(attrgetter("name"), self.get_params())
#        layer_shapes = map(lambda w: w.shape, self.get_weights())
#        return zip(layer_weight_names, layer_weights, layer_shapes)



class Model( object ):
    """generic model class with basic functionality"""

    #__metaclass__ = abc.ABCMeta

    def __init__(self, layers):
        self.__layers = layers
        log.info("declared model\n%s", str(self))

    def __getitem__(self, i):
        return self.__layers[i]

    def __len__(self):
        return len(self.__layers)

    def __setitem__(self, i, it):
        self.__layers[i] = it

    def __delitem__(self, i):
        del self.__layers[i]

    @staticmethod
    def weight_path(i, wname, dir=None):
        """path to a weight directory

        :param dir: path to the training instance.
                    if None a relative path will be returned
        :param int i: index of the layer
        :param str wname: weight name. equals the theano.tensor name
        :rtype: str"""

        p = os.path.join("layer_%d" % i, wname)
        if not (dir is None):
            p = os.path.join(dir, p)
        return p

    def load(self, path):
        """load model weights

        :param str path: the path to the training location. i.e., a.h5:trainid
        """

        arch, trpath = archive.split(path)
        for i, layer in enumerate(self):
            layer_weights = []
            for wn in map(attrgetter("name"), layer.get_params()):
                wpath = self.weight_path(i, wn, trpath)
                layer_weights.append(archive.load_object(arch, wpath).values)
            layer.set_weights( layer.load_flat_weights(layer_weights) )
            log.info("updating weights of %dth layer (%s)", i, layer)

    def save(self, path):
        """save model weights

        :param str path: the path to the training location.
            i.e., a.h5:dataset/trainid
        """

        def weights_names_shapes(layer):
            "[(wname, w_flat, w_shape), ...]"

            layer_weights = layer.get_flat_weights()
            layer_weight_names = map(attrgetter("name"), layer.get_params())
            layer_shapes = map(lambda w: w.shape, layer.get_weights())
            return zip(layer_weight_names, layer_weights, layer_shapes)

        log.info("saving model to %s", path)
        arch, trpath = archive.split(path)
        for i, layer in enumerate(self):
            for wn, w, wsh in weights_names_shapes(layer):
                archive.save_object(arch,
                                    self.weight_path(i, wn, trpath),
                                    pd.Series(w))
                archive.save_object(arch,
                                    self.weight_path(i, "%s_shape" % wn, trpath),
                                    pd.Series(wsh, index=range(len(wsh))))

    def weight_norm(self, degree):
        """compute the L1 or L2 norm of weights

        :param degree: keyword `l1` or `l2`
        :rtype: the L1 or L2 norm of weights (float)"""

        return sum( map(lambda l: l.weight_norm(degree), self) )

    def get_weights(self):
        """numpy arrays for the weights

        :rtype: the list of parameters of layers
        """

        return map(lambda l: l.get_weights(), self.__layers)

    def set_weights(self, wlst):
        for w, i in enumerate(wlst):
            self.__layers[i].set_weights( w )

    def get_params(self):
        """(theano) symbolic variables for the weights

        :rtype: the list of parameters of layers
        """

        return map(lambda l: l.get_params(), self.__layers)

    def __str__(self):
        return "\n".join( map(str, self) )


def verbose_compile(func):
    """decorator that explains what is being theano-compiled,
    by logging the functions __doc__"""

    def newfunc(*args, **kwargs):
        log.info( func.__doc__ )
        return func(*args, **kwargs)
    return newfunc


def alloc_shared_weights_(shape, dtype, name, fan_in, rng):
    """alloc a matrix of weights and return a theano shared variable
    Note : optimal initialization of weights is dependent on the
    activation function used (among other things). Here  I replicate
    http://www.deeplearning.net/tutorial/mlp.html#mlp
    """

    if type(rng) == float:
        val = str(rng)
        var = theano.shared(value=np.zeros(shape, dtype=dtype) + rng, name=name)
    elif type(rng) == np.random.RandomState:
        val = "rnd"
        thr = np.sqrt(3. / fan_in)
        var = theano.shared(value=np.asarray(rng.uniform(low=-thr, high=thr,
                            size=shape), dtype=dtype), name=name)
    else:
        raise ValueError("cannot understant `rng`")

    log.debug("%s (%s) = %s (%s)", str(var), str(var.get_value().shape),
              val, str(var.get_value().dtype))

    return var


def adjust_lr(err, lrmax):
    "adjust error rate for the next batch"

    ## another strategy is to keep the learning rate fixed
    ## if it does not increase the error
    ## and to sqrt(half) it otherwise

    if len(err) < 2:
        return lrmax
    E = max( err[-2], err[-1] )
    if E == 0:
        return lrmax
    if (err[-2] - err[-1]) / E < -0.001:  # went up by more than 1%
        new_err = [lrmax * np.sqrt((len(err) - 1) / float(len(err)))]
        return float( np.array(new_err, dtype=np.float32)[0] )
    return lrmax


from nccn import LogisticReg, LinearRegression


class ProblemType(object):
    """class that emulates an enum{classification, regression}"""

    classification = 1
    regression = 2
    choices = ("C", "R")

    top_layer = {"C": LogisticReg, classification: LogisticReg,
                 "R": LinearRegression, regression: LinearRegression}

    @classmethod
    def parse(cls, ptp):
        """parse a char/int into one of ProblemType.classification and
        ProblemType.regression

        :param ptp: char or int to be parsed
        :rtype: one of ProblemType.classification or ProblemType.regression"""

        if ptp in (cls.classification, cls.regression):
            return ptp
        return {"C": cls.classification, "R": cls.regression}[ptp]

    @classmethod
    def is_valid(cls, pt):
        """wherther or not ProblemType.parse will succeed

        :param pt: char or int to be parsed
        :rtype: boolean"""

        return (pt in (1, 2)) or (pt in cls.choices)

    @classmethod
    def ds_out(cls, ds, pt):
        """extract the correct output variable from ds based on problem type

        :param ds: dataset instance
        :param pt: char or int to be parsed
        :rtype: boolean"""

        if pt in ("C", 1):
            return ds.T
        return ds.Y

    @classmethod
    def ds_shout(cls, ds, pt):
        """extract the correct *theano shared* output variable from
        ds based on problem type

        :param ds: dataset instance
        :param pt: char or int to be parsed
        :rtype: theano shared tensor"""

        if cls.parse(pt) == cls.classification:
            return ds.shT
        return ds.shY

    @classmethod
    def ds_nout(cls, ds, pt):
        """extract the correct dimension output variable from
        ds based on problem type

        :param ds: dataset instance
        :param pt: char or int to be parsed
        :rtype: int"""

        if cls.parse(pt) == cls.classification:
            return ds.labels
        return 1

    @classmethod
    def cnn_lmon(cls, pt):
        if cls.parse(pt) == cls.classification:
            return monitor.CnnLearnMonitorC
        return monitor.CnnLearnMonitorR
