"""
utilities for commandline argument parsing
"""

import os
import argparse
import re
from dimer import archive

class ArgTypes(object):
    @staticmethod
    def is_file(path):
        """check that path is a file and return its absolute path

        :param str path: path to check
        :rtype: absolute path or error"""

        if not os.path.isfile(path):
            raise IOError("%s is not a file. pipes/symlinks won't work" % path)
        return os.path.abspath(path)

    @staticmethod
    def is_dsarch(s):
        """a checker for correct specification of path to a dataset

        :param str path: path to the dataset (path_to_file:dsname)
        :rtype: the same path if correct, raises error otherwise"""

        try:
            return archive.dset_path(s)
        except AttributeError:
            raise AttributeError("%s in wrong format. %s" % (s,
                                 archive.DSPEC_MSG))

    @staticmethod
    def is_dspath(s):
        return  ArgTypes.is_dsarch(s)

    @staticmethod
    def is_pint(s):
        i = int(s)
        if i > 0:
            return i
        raise ValueError("%s is not a positive integer" % s)

    @staticmethod
    def is_uint(s):
        i = int(s)
        if i >= 0:
            return i
        raise ValueError("%s is not a non-negative integer" % s)

    @staticmethod
    def is_chrom(s, P=re.compile("chro?m?\d\d?")):
        if P.match(s):
            return s
        raise ValueError("%s doesn't seem to be a chromosome name" % s)

    @staticmethod
    def is_emptydir(path):
        ap = os.path.abspath(path)
        if not os.path.isdir(ap):
            raise IOError("%s (originally %s) is not a directory" % (ap, path))
        if os.listdir(ap):
            raise IOError("%s (originally %s) is not empty" % (ap, path))
        return ap


def parser_inst(desc=__doc__):
    pinst = argparse.ArgumentParser(
        description=desc,
        formatter_class=argparse.ArgumentDefaultsHelpFormatter,
        epilog="Olgert Denas (odenas@emory.edu)")
    return pinst


def attach_config(cls, pinst):
    gr = pinst.add_argument_group(title=cls._section,
                                  description="Model configuration")
    attr_help = cls._help if hasattr(cls, "_help") else ("NA",) * len(cls._fields)
    for nm, tp, hlp in zip(cls._fields, cls._types, attr_help):
        gr.add_argument("--%s" % nm, type=tp, help=hlp)


def common_frepr_opts(parser, set_odir=True):
    """attach commonly used options in computation of hidden representations
    from a stack of dAEs

    :param argparse.ArgumentParser parser: argparser to attach opts to
    :param bool set_odir: enforce out-dir to be empty
    :rtype: argparse.ArgumentParser parser with options attached"""

    uint = ArgTypes.is_pint(i)
    parser.add_argument("input", type=ArgTypes.is_dsarch,
                        help="Input archive. " + archive.DSPEC_MSG)
    parser.add_argument("trid", type=str, help="Train id")
    parser.add_argument("slice_size", type=is_slice, help="per-cpu genome region")
    parser.add_argument("output", type=is_outdir if set_odir else str,
                        help="Output dir")
    parser.add_argument("chrom", type=ArgTypes.is_chrom, help="chromosome")
    parser.add_argument("layer", type=uint,
                        help="metalabels are repr. of this layer")

    parser.add_argument("--start", type=ArgTypes.is_uint(i), default=0, help="start")
    parser.add_argument("--end", type=uint, default=0, help="end")
    parser.add_argument("--batch_size", type=uint, default=700,
                        help="batch size")
    parser.add_argument("--bin_size", type=uint, default=10,
                        help=("binning size. this should match the bin-size "
                              "used to create the data-archive"))
    parser.add_argument("--bin_op", choices=("mean", "sum"), default="mean",
                        help="binning operation")
    parser.add_argument("--par", type=uint, default=1,
                        help="parallelize")
    parser.add_argument("--signal_dir", type=str, default=".",
                        help="directory of bigwig files")
    parser.add_argument("--dry_run", action='store_true', default=False,
                        help="Dry run")

    return parser

