%------------------------------------------------------------------------
% How to use this algorithm?
%
%   We need to initialize some values.
%   Here since we implement the Merton's model for probability
%   measure, we have,
%       Initial conditions: h(x) = (1 - e^x)+;
%                           g(tao, x) = 0;
%       Probability measure:v(x) = 0.1 * exp(-x .* x * 0.5) / sqrt(2*pi)
%------------------------------------------------------------------------


%   1.
%   Initial values
%           Ao: the order of truncated upper bound on x of the grid
%           T: maturity date
%           M: the number of subintervals of time
%           N: the number of subintervals of x movements
%           sigma: the volatility of the underlying stock
%           r: risk-free rate
%           hx: vector of initial conditions

Ao = 10;
T = 1;
M = 80;
N = 100;
sigma = 0.15;
r = 0.05;

S0 = 100;
K = 100;

myFun = @(x) x .* x .* (0.1 * exp(-x .* x * 0.5) / sqrt(2*pi));

A = Ao * sqrt(T * (sigma * sigma + quadgk(myFun,-inf, inf)));

dx = 2 * A / N;
dt = T / M;

% Initial conditions
h = @(x) K - S0 * exp(x);
%h = @(x) 1 - exp(x);
hx = zeros(N+1,1);
x = zeros(N+1,1);
t = linspace(0,T,M+1)';

for i = 1:(N+1)
    x(i) = -A + i * dx;
    hx(i) = max(0, h(x(i)));

    if hx(i) < 0
        hx(i) = 0;
    end
end

%   2.
%   localization conditions
%           ep: given tolerance
%           Bl: left hand truncation of the integral
%           Br: right hand truncation of the integral
%   Without losing generality, Bl < -1, Br > 1, and
%       ep needs to be smaller than 1/sqrt(2*pi).
%
ep = 1e-3;
Bl = -sqrt(-2*log(10*ep*sqrt(2*pi)));
Br = - Bl;
%   Since [Bl, Br] is part of [x0 + (Kl - 0.5)dx, x0 + (Kr + 0.5)dx],
Kl = max(floor(Bl / dx + 0.5),-N);
Kr = - Kl;

%   3.
%   Simulate vectors u with function BuildU.m
[u, C] = BuildU( A, T, M, N, Kl, Kr, sigma, r, hx,S0,K );

%   4.
%   Calculate the expected price of C.Price is desired calculated price.
%   Visualize the solutions
sumC = sum(sum(C));
Price = sumC / N / M
x1 = linspace(-A,A,N+1)'+K;
plot(x1,u(:,1),'-',x1,u(:,2),'--', x1, u(:,M-2),'o-')
legend('t=T', 't=T-1','t=2')
xlabel('S'), ylabel('Payoff')