function [ Levy, BS ] = LevyMerton...
    ( Ao, T, M, N, S0, K, sigma, r, CallOrPut, hFun)
%
%   This function takes inputs initialized by the user, and uses
%   an implicit-explicit finite difference scheme of Levy process
%   to calculate the price of call; also gives a comparison to 
%   the Black-Scholes
%   model's result.
%
%   Input
%           Ao: the order of truncated upper bound on x of the grid
%           T: maturity date
%           M: the number of subintervals of time
%           N: the number of subintervals of x movements
%           sigma: the volatility of the underlying stock
%           r: risk-free rate
%           hx: vector of initial conditions
%           S0: initial stock price
%           K: strike price
%           CallOrPut: 0 for Call, 1 for Put
%           hFun: initial conditions, if not passing hFun, default
%                   will be h(x) = (1 - e^x)+
%   Output
%           Levy: our result by the Levy processes with Merton's model
%           BS: result by the Black-Scholes model

%------------------------------------------------------------------------
% How to use this algorithm?
%
%   We need to initialize some values.
%   Here since we implement the Merton's model for probability
%   measure, we have,
%       Initial conditions: h(x) = (1 - e^x)+; (by default)
%                           g(tao, x) = 0;
%       Probability measure:v(x) = 0.1 * exp(-x .* x * 0.5) / sqrt(2*pi)
%------------------------------------------------------------------------

%   1.
%   Initial values

myFun = @(x) x .* x .* (0.1 * exp(-x .* x * 0.5) / sqrt(2*pi));

A = Ao * sqrt(T * (sigma * sigma + quadgk(myFun,-inf, inf)));

dx = 2 * A / N;
dt = T / M;

if nargin == 9
    if CallOrPut == 0
        hFun = @(x) exp(x) - 1;
    else if CallOrPut == 1
        hFun = @(x) 1 - exp(x);
        end
    end
end
hx = zeros(N+1,1);
x = zeros(N+1,1);
t = linspace(0,T,M+1)';

for i = 1:(N+1)
    x(i) = -A + i * dx;
    hx(i) = max(0, hFun(x(i)));

    if hx(i) < 0
        hx(i) = 0;
    end
end

%   2.
%   localization conditions
%           ep: given tolerance
%           Bl: left hand truncation of the integral
%           Br: right hand truncation of the integral
%   Without losing generality, Bl < -1, Br > 1, and
%       ep needs to be smaller than 1/sqrt(2*pi).
%
ep = 1e-3;
Bl = -sqrt(-2*log(10*ep*sqrt(2*pi)));
Br = - Bl;
%   Since [Bl, Br] is part of [(Kl - 0.5)dx, (Kr + 0.5)dx],
Kl = max(floor(Bl / dx + 0.5),-N);
Kr = - Kl;

%   3.
%   Simulate vectors u with function BuildU.m
[u, C] = BuildU( A, T, M, N, Kl, Kr, sigma, r, hx,K,S0 );

%   4.
%   Calculate the expected price of C.Price is desired calculated price.
sumC = sum(sum(C));
Levy = sumC / N / M;
d1 = (log(S0/K) + (r + sigma*sigma / 2) * T) / (sigma * sqrt(T));
d2 = d1 - sigma * sqrt(T);
PV_K = exp(-r*T) * K;
if CallOrPut == 0
    BS = S0 * normcdf(d1,0,1) - PV_K * normcdf(d2,0,1);
else if CallOrPut == 1
        BS = PV_K * normcdf(-d2,0,1) - S0 * normcdf(-d1,0,1);
    end
end
end