function [ J, alpha, lambda ] = BuildJ( N, dx, dt, Kl, Kr )
%   function BuildJ
%
%   This algorithm build the matrix J which we need for the
%   right-hand side computation. Here, we use the Merton's
%   model to construct the jump steps function v:
%
%          vFun = @(x) 0.1 * exp(-x .* x * 0.5) / sqrt(2*pi);
%
%   We then calibrate the matrix J into the complete right-
%   hand side form: (I + dt * J).
%
%   Input:  N: the number of subintervals of x
%           dx: movement step size
%           dt: time step size
%
%   Output: J: (N+1)x(N+1) matrix of the right-hand side
%             information to update the vector u_n to u_n+1
%           alpha: coefficient for localization of boundaries
%

vFun = @(x) 0.1 * exp(-x .* x * 0.5) / sqrt(2*pi);

v = zeros(Kr - Kl + 1, 1);
lambda = 0;
alpha = 0;
J = zeros(N+1,N+1);

for i = Kl : Kr
    lower = (i - 0.5) * dx;
    upper = (i + 0.5) * dx;
    k =i + (1 - Kl);
    y = Kl * dx;
    v(k) = quad(vFun, lower, upper);
    lambda = lambda + v(k);
    alpha = alpha + (exp(y)-1)*v(k);
    V = ones(N+1-abs(i),1) * v(k);
    J = J + diag(V, i);
end

J = eye(N+1, N+1) + dt * J;
end